/******************************************************************************
*                                                                             *
* License Agreement                                                           *
*                                                                             *
* Copyright (c) 2009 Altera Corporation, San Jose, California, USA.           *
* All rights reserved.                                                        *
*                                                                             *
* Permission is hereby granted, free of charge, to any person obtaining a     *
* copy of this software and associated documentation files (the "Software"),  *
* to deal in the Software without restriction, including without limitation   *
* the rights to use, copy, modify, merge, publish, distribute, sublicense,    *
* and/or sell copies of the Software, and to permit persons to whom the       *
* Software is furnished to do so, subject to the following conditions:        *
*                                                                             *
* The above copyright notice and this permission notice shall be included in  *
* all copies or substantial portions of the Software.                         *
*                                                                             *
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR  *
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,    *
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE *
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER      *
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING     *
* FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER         *
* DEALINGS IN THE SOFTWARE.                                                   *
*                                                                             *
* This agreement shall be governed in all respects by the laws of the State   *
* of California and by the laws of the United States of America.              *
* Altera does not recommend, suggest or require that this reference design    *
* file be used in conjunction or combination with any other product.          *
******************************************************************************/


/**********************************************************************
 *
 * Filename:    main.c
 * Author:      JCJB
 * Date:        09/16/2009
 * 
 * Version 1.0 (Initial release)
 * 
 * Description:
 * 
 * This software uses the modular SGDMA to move contents from one
 * location in memory to another in the same memory.  The transfer
 * is comprised of multiple buffers and uses an interrupt to determine
 * when the last buffer has been written to the destination location.
 * The test is portable to other designs and memory with user
 * selected settings located below.  Since this design performs
 * destructive accesses you must ensure that the memory under test
 * does not contain any Nios II sections (.text, .rodata, etc...)
 * 
 * The software performs the following steps:
 * 
 * 1)  Generates a list of buffer locations for the source and
 *     destination.  The buffer lengths are either fixed or generated
 *     randomly.
 * 
 * 2)  Source buffers are populated with an incrementing pattern of
 *     0, 1, 2, 3, etc....
 * 
 * 3)  Data cache is flushed and the benchmark timer is started.
 * 
 * 4)  Descriptors are created and sent to the modular SGDMA.
 * 
 * 5)  Software waits for the transfer complete interrupt to occur.
 * 
 * 6)  Benchmark timer is read.
 * 
 * 7)  Destination data is verified for accuracy.  If the destination
 *     is corrupt an error message is output to the console and the 
 *     software terminates.
 * 
 * 8)  Benchmark data is output to the console and the software returns
 *     to step 1 if there are more tests to perform.
 **********************************************************************/

#include "system.h"
#include "stdio.h"
#include "sgdma_dispatcher.h"
#include "descriptor_regs.h"
#include "csr_regs.h"
#include "stdlib.h"
#include "sys/alt_irq.h"
#include "sys/alt_timestamp.h"
#include <sys/alt_cache.h>
#include "sys/alt_stdio.h"

#include "sll_psram_tests.h"
#include "sll_ca_xspi_mc_init.h"
#include "sll_drvr_emlx.h"
#include "sll_flash_test_003.h"


//------------------------------------------------------------------
//Device on J1 (FIXED configuration) -
//
// Only One has to be uncommented
//------------------------------------------------------------------
#define EM064LX_DEV

#define SLL_J1_CS0_DEVICE  MEM_EM064LX_X1
#define SLL_J1_CS1_DEVICE  MEM_NONE

//------------------------------------------------------------------
// Generic Device Parameter on Channel J3
//------------------------------------------------------------------
#define SLL_J1_MEM_CTRL     (SLL_XSPI_MC_AVMM_TOP_0_IAVS_REG_BASE)
#define SLL_J1_MEM_BASE     (SLL_XSPI_MC_AVMM_TOP_0_IAVS0_BASE)
#define SLL_J1_MEM_SPAN     (SLL_XSPI_MC_AVMM_TOP_0_IAVS0_SPAN)


//------------------------------------------------------------------
// SDRAM Memory
//------------------------------------------------------------------
#define SDRAM_MEM_BASE     (SDRAM_BASE)


//------------------------------------------------------------------
// SDRAM Span Test parameter
//------------------------------------------------------------------
#define SDRAM_TEST_BUFFER_SIZE (1024*1024)

//------------------------------------------------------------------
// fill parameters for g_xSPI_MBMC_J1 instantiation 
//------------------------------------------------------------------
xSPI_MBMC_Info g_xSPI_MBMC_J1 = {
	  .csr_base      =  SLL_J1_MEM_CTRL,
    /*Device 0 parameters*/
    .dev0_name     =  SLL_J1_CS0_DEVICE,
    .dev0_mem_base =  SLL_J1_MEM_BASE,
    .dev0_mem_size =  SLL_J1_MEM_SPAN,
    
    /*Device 1 parameters*/
    .dev1_name     =  SLL_J1_CS1_DEVICE
};


//------------------------------------------------------------------
// Initialise Flash handle doe Everspin EM06LX Device
//------------------------------------------------------------------

SLL_EMLX_FLASH_INSTANCE      (Flash_J1);

//------------------------------------------------------------------
// Programming Parameters
//------------------------------------------------------------------
#define SLL_FLASH_PROGRAMMING_OFFSET (1024*1024)
#define SLL_FLASH_PROGRAMMING_SIZE   (1024*1024)


/********************************************************
 * Test modifications starts here                       *
 ********************************************************/
/* If you port this design to a different hardware platform set the following two value for the memory under test.
   Make sure you don't place your code in this test memory since it's a destructive test.
 */
  
#define RAM_SPAN  (0x00800000)


// These values can be changed to modify the test
#define MAXIMUM_BUFFER_SIZE (16*1024)     // 16kB maximum buffer size (hardware is setup to handle less than 1MB)
#define RANDOM_BUFFER_LENGTH_ENABLE 0     // set to zero to force all buffers to be set to a length of MAXIMUM_BUFFER_SIZE
#define NUMBER_OF_BUFFERS 128             // must be a multiple of 2 since the buffers must be aligned to 4 byte boundaries
#define NUMBER_OF_TESTS 10                // this value is only applicable when 'INFINITE_TEST' is set to 0
#define INFINITE_TEST 0                   // when 0 the test will only run 'NUMBER_OF_TESTS' times

/********************************************************
 * Test modifications ends here                         *
 ********************************************************/


// each buffer start location will be evenly spaced apart so there could be gaps between the end of one buffer and the start of another
#define ADDRESS_STEP_SIZE ( (RAM_SPAN / 2) / NUMBER_OF_BUFFERS )

#if(((NUMBER_OF_BUFFERS * 2) * MAXIMUM_BUFFER_SIZE) > RAM_SPAN)
  #error There is not enough test memory for the test parameter you have set, either reduce 'NUMBER_OF_BUFFERS' or 'MAXIMUM_BUFFER_SIZE'
#endif

// chances are the error above will be issued as well if the following condition is true but just in case....
#if (MAXIMUM_BUFFER_SIZE > ADDRESS_STEP_SIZE)
  #error The maximum buffer size is too large for the number of buffers under test.  Either reduce 'NUMBER_OF_BUFFERS' or 'MAXIMUM_BUFFER_SIZE'
#endif


// some defines for the time that this file was compiled which will be used to seed random number generation.  If you recompile and this file is not changed then you'll end up with the same seed (run a clean on the project to get a new seed)
#define SECONDS (((__TIME__[6] - 48) * 10) + (__TIME__[7] - 48))
#define MINUTES (((__TIME__[3]- 48) * 10) + (__TIME__[4] - 48))
#define HOURS (((__TIME__[0] - 48) * 10) + (__TIME__[1] - 48))
#define SEED ((SECONDS + MINUTES + HOURS) + (SECONDS ^ MINUTES) + (MINUTES ^ HOURS) + (SECONDS << 16))  // picked this formula at random...

#define TERMINAL_KILL_CHARACTER 0x4




/******************************************************************
*  Function: WaitAnyKey
*
*  Purpose: Parses an input string for the character '\n'.  Then
*           returns the string, minus any '\r' characters it 
*           encounters.
*
******************************************************************/
void WaitAnyKey(void)
{
  char ch = ' ';

  lld_printf("Press any key to continue : \n");
  ch = alt_getchar();

}

// generates a seed for the random number generator based on the date and time of the compilation
void random_seed()
{
  lld_printf("Compiled at %d:%d:%d\n\n", HOURS, MINUTES, SECONDS);
  lld_printf("Using 0x%x as the seed\n", SEED);
  srand((unsigned int)SEED);        
}



// flag used to determine when all the transfers have completed
volatile int sgdma_interrupt_fired = 0;

static void sgdma_complete_isr (void *context)
{
  sgdma_interrupt_fired = 1;
  clear_irq (MODULAR_SGDMA_DISPATCHER_CSR_BASE);
}



//-----------------------------------------------------------
//DMA Test
//-----------------------------------------------------------
int sgdma_test(uint32_t source_base, uint32_t dest_base, uint32_t memory_size, uint32_t *bandwidth, int skip_verify) {
	
  sgdma_standard_descriptor a_descriptor;
  sgdma_standard_descriptor * a_descriptor_ptr = &a_descriptor;  // using this instead of 'a_descriptor' throughout the code
  unsigned long length[NUMBER_OF_BUFFERS];
  unsigned long read_address[NUMBER_OF_BUFFERS];
  unsigned long write_address[NUMBER_OF_BUFFERS];
  unsigned long i, j, test_counter, data_read, control_bits, total_length_count;
  unsigned char * ram_access_ptr;  // using this to step through the buffers one byte at a time
  unsigned long transfer_time, test_throughput;


  lld_printf( "Altera's Modular SGDMA memory copy benchmarking with integrity verification.\n" );

  lld_printf( "DATA_SOURCE_BASE:            0x%08x\n", source_base );
  lld_printf( "DATA_DESTINATION_BASE:       0x%08x\n", dest_base );
  lld_printf( "MAXIMUM_BUFFER_SIZE:         %u\n", MAXIMUM_BUFFER_SIZE );
  lld_printf( "RANDOM_BUFFER_LENGTH_ENABLE: %u\n", RANDOM_BUFFER_LENGTH_ENABLE );
  lld_printf( "NUMBER_OF_BUFFERS:           %u\n", NUMBER_OF_BUFFERS );
  lld_printf( "NUMBER_OF_TESTS:             %u\n", NUMBER_OF_TESTS );

  /******************************
  Start of DMA Test
  ******************************/

  if(RANDOM_BUFFER_LENGTH_ENABLE == 1)
  {
    random_seed();    // seed the random number generator
  }

  test_counter = 0;

  /*  Testing starts here  */
  do {

    total_length_count = 0;

    // generate buffer base addresses and lengths
    for(i = 0; i < NUMBER_OF_BUFFERS; i++)
    {
      read_address[i]  = source_base + (i * ADDRESS_STEP_SIZE);
      write_address[i] = dest_base   + (i * ADDRESS_STEP_SIZE);
      if (RANDOM_BUFFER_LENGTH_ENABLE == 1)
      {
        length[i] = 1 + ((unsigned long)(((double)rand() / (double)RAND_MAX) * (double)MAXIMUM_BUFFER_SIZE));  // 1 + (random number [0,1) * max buffer length)
      }
      else
      {
        length[i] = (test_counter+1) * 1024; //MAXIMUM_BUFFER_SIZE;
      }
      total_length_count += length[i];
    }  
  
    // populate the source data
    for(i = 0; i < NUMBER_OF_BUFFERS; i++)
    {
      ram_access_ptr = (unsigned char *)read_address[i];
      for(j = 0; j < length[i]; j++)
      {
        *ram_access_ptr++ = (unsigned char)(j & 0xFF);  // writing a repeating pattern of 0, 1, ... 0xFF, 0, 1, ... 0xFF, etc..
      }  
    }
    
    alt_dcache_flush_all(); // need to make sure all the data is written out to memory

    if (alt_timestamp_start() != 0)
    {
      lld_printf("Failed to start the timer, make sure the timestamp timer is set.%c", TERMINAL_KILL_CHARACTER);
      return 0;
    }
  
    // start writing descriptors to the SGDMA
    for(i = 0; i < NUMBER_OF_BUFFERS; i++)
    {
      while ((RD_CSR_STATUS(MODULAR_SGDMA_DISPATCHER_CSR_BASE) & CSR_DESCRIPTOR_BUFFER_FULL_MASK) != 0) {}  // spin until there is room for another descriptor to be written to the SGDMA
  
      control_bits = (i == (NUMBER_OF_BUFFERS-1))? DESCRIPTOR_CONTROL_TRANSFER_COMPLETE_IRQ_MASK : DESCRIPTOR_CONTROL_EARLY_DONE_ENABLE_MASK;  // go bit is handled 'construct_standard_mm_to_mm_descriptor'

      construct_standard_mm_to_mm_descriptor (a_descriptor_ptr, (alt_u32 *)read_address[i], (alt_u32 *)write_address[i], length[i], control_bits);
  
      if(write_standard_descriptor (MODULAR_SGDMA_DISPATCHER_CSR_BASE, MODULAR_SGDMA_DISPATCHER_DESCRIPTOR_SLAVE_BASE, a_descriptor_ptr) != 0)
      {
        lld_printf("Failed to write descriptor 0x%lx to the descriptor SGDMA port.%c", i+1, TERMINAL_KILL_CHARACTER);
        return 1; 
      }
    }
    
    while (sgdma_interrupt_fired == 0)  {}  // keep spinning until the interrupt fires when the last word is written to the destination location by the SGDMA

    transfer_time = alt_timestamp();  // number of clock ticks from the time that descriptors where formed and sent to the SGDMA to the time of the last memory write occuring
      
    // verify the data written by the SGDMA at the destination addresses
    if (skip_verify == 0) {

       for(i = 0; i < NUMBER_OF_BUFFERS; i++)
       {
         ram_access_ptr = (unsigned char *)write_address[i];
         for(j = 0; j < length[i]; j++)
         {
           data_read = *ram_access_ptr++;  // should be a repeating pattern of 0, 1, ... 0xFF, 0, 1, ... 0xFF, etc..
           if ((j & 0xFF) != data_read)
           {
             if (INFINITE_TEST == 1)  // infinite test so we don't know what test we are on
             {
               lld_printf("Buffer %ld, base 0x%lx offset 0x%lx failed, expected 0x%lx but read 0x%lx.%c", (i+1), write_address[i], j, (j&0xFF), data_read, TERMINAL_KILL_CHARACTER);
             }
             else
             {
               lld_printf("Test %ld buffer %ld, base 0x%lx offset 0x%lx failed, expected 0x%lx but read 0x%lx.%c", (test_counter+1),(i+1), write_address[i], j, (j&0xFF), data_read, TERMINAL_KILL_CHARACTER);
             }
             return 1;  
           }
         }
       }
    }
  
    sgdma_interrupt_fired = 0;  // set back to 0 to perform another test
    test_counter++;
    // [#ticks for test] / [#timer ticks/s] = total test time in seconds
    // throughput = [total data] / [total time in seconds] = ([total data] * [#timer ticks/s]) / [#ticks]  <--- this would be bytes per second so divide by 1024*1024 to get MB/s
    test_throughput = (unsigned long)((((double)total_length_count) * ((double)alt_timestamp_freq())) / ((double)transfer_time * 1024 * 1024) );
    
    *bandwidth = (test_throughput*2);
    if(INFINITE_TEST == 1)
    {
      lld_printf("Test completed for %ld Kilobytes - Avg memory copy throughput is %ld MBytes/s - Avg Read/Write throughput is ~%ld MBytes/s.\n", (total_length_count/1024), test_throughput, test_throughput*2);
    }
    else
    {
      lld_printf("Test number %02ld completed for %ld Kilobytes - Avg memory copy throughput is %ld MBytes/s - Avg Read/Write throughput is ~%ld MBytes/s.\n", (test_counter), (total_length_count/1024), test_throughput, test_throughput*2);
    }
    
  } while ((INFINITE_TEST == 1) | (test_counter < NUMBER_OF_TESTS));  // end of test loop

  return 1;  
  
}



int main()
{

  int sdram_error = 0;
  int J1_mem_error = 0;
  uint32_t sgdma_bandwidth_1 = 0;
  
  flash_timing_results J1_results;
  
  /******************************
   Select Device Populated on Board
  ******************************/
  uint32_t source_base, dest_base, memory_size;
		

  alt_ic_isr_register (MODULAR_SGDMA_DISPATCHER_CSR_IRQ_INTERRUPT_CONTROLLER_ID, MODULAR_SGDMA_DISPATCHER_CSR_IRQ, sgdma_complete_isr, NULL, NULL);  // register the ISR
  enable_global_interrupt_mask(MODULAR_SGDMA_DISPATCHER_CSR_BASE);                // turn on the global interrupt mask in the SGDMA

  //----------------------------------------------
  //Create buffer in SRAM
  //----------------------------------------------
  uint32_t *sdram_buffer_ram = 0;

	sdram_buffer_ram = memalign(SDRAM_TEST_BUFFER_SIZE, SDRAM_TEST_BUFFER_SIZE); //assign 1 Mbyte, aligned to 1 Mbyte
  
	//check if buffer is allocated
	if (sdram_buffer_ram == NULL)
	{
		printf("Cannot allocate memory buffer\r\n");
		free(sdram_buffer_ram);
    
    while (1) {
    //stop here
    }
    
	} else {
		printf("SDRAM memory buffer allocated at address 0x%x\r\n", sdram_buffer_ram);
	}


  //-------------------------------------------------------- 
  // Prepare SLL xSPI-MBMC Core 0  (mapped to Cruvi adapter on J1)
  //
  // SLL xSPI-MBMC Core 0 (on FPGA) is configured in static mode
  // The core needs to be initialised according to the device
  // present on the board.
  //
  // In this case, we are selecting Everspin EM064LX device
  // 
  //-------------------------------------------------------- 
//  xspi_mc_cntrl_init(&g_xSPI_MBMC_J1);


  sll_nor_flash_dev_int(&g_xSPI_MBMC_J1, &Flash_J1);

  
  //-------------------------------------------------------- 
  //Start with SDRAM Test 
  //-------------------------------------------------------- 
  lld_printf ("Testing SDRAM \n");

  lld_printf ("\n .... Nios II Read/write tests\n");
  sdram_error = sll_psram_tests_all(sdram_buffer_ram, SDRAM_TEST_BUFFER_SIZE, 0, 1, 1);

  //-------------------------------------------------------- 
  //Start with Test on J3
  //-------------------------------------------------------- 
  lld_printf ("Testing Everspin EM064LX device in PSRAM Mode \n");

  //------------------------------------------------------------
  // Nios 32-bit read-write Cached/Uncached tests
  //------------------------------------------------------------
  lld_printf ("\n .... Nios II Read/write tests\n");
  J1_mem_error += sll_psram_tests_read_write_span_accessibility_test(g_xSPI_MBMC_J1.dev0_mem_base, g_xSPI_MBMC_J1.dev0_mem_size, 1 ); 
  WaitAnyKey();  
  
  //------------------------------------------------------------
  // SGDMA Test
  //------------------------------------------------------------
  lld_printf (" \n .... SGDMA Test on EM064LX \n");
  /*Set Source / Destination Base */
	source_base =	g_xSPI_MBMC_J1.dev0_mem_base;
	memory_size =	g_xSPI_MBMC_J1.dev0_mem_size;
  dest_base   = source_base + (memory_size/2); 
  
  sgdma_test(source_base, dest_base, memory_size, &sgdma_bandwidth_1, 0) ;  
  WaitAnyKey();  

  lld_printf("--------------------------------------\r\n");
  lld_printf ("Testing Everspin EM064LX device in Flash Mode \n");
  lld_printf("--------------------------------------\r\n");
  J1_mem_error += sll_flash_test_003( &Flash_J1, sdram_buffer_ram, SLL_FLASH_PROGRAMMING_OFFSET, SLL_FLASH_PROGRAMMING_SIZE, &J1_results);
   WaitAnyKey();  

  lld_printf("--------------------------------------\r\n");
  lld_printf(" Nios II and Avalon Ports running at %0d Mhz \r\n", (ALT_CPU_CPU_FREQ/(1000*1000)));
  lld_printf("--------------------------------------\r\n\r\n");

  lld_printf("--------------------------------------\r\n");
  lld_printf ("Test Results : Read    ->  Write     DMA Bandwidth \n");
  lld_printf ("Device       : EM064LX     EM064LX     %0d  Mbytes/s  \n", sgdma_bandwidth_1);
  lld_printf("--------------------------------------\r\n\r\n");

  lld_printf("--------------------------------------\r\n");
  lld_printf ("Test Results            :  Data Size %0d Kbytes \n", SLL_FLASH_PROGRAMMING_SIZE/1024);
  lld_printf ("                        :  EM064LX  \r\n");
  lld_printf ("Erase Time (ms)         :     %0d     \n", J1_results.flash_erase_time);
  lld_printf ("Erase Verify Time (ms)  :     %0d     \n", J1_results.flash_erase_verify_time);
  lld_printf ("Program Time(ms)        :     %0d     \n", J1_results.flash_program_time);
  lld_printf ("Program Verify Time(ms) :     %0d     \n", J1_results.flash_program_verify_time);
  lld_printf("--------------------------------------\r\n\r\n");
  lld_printf("All tests have completed.%c", TERMINAL_KILL_CHARACTER);
 return 0;

}
